<?php
ini_set('display_errors',1);
error_reporting(E_ALL);

/* Copyright (C) 2013-2022  Laurent Destaileur        <ely@users.sourceforge.net>
 * Copyright (C) 2014     Regis Houssin               <regis.houssin@inodbox.com>
 * Copyright (C) 2024-2025 MDW                        <mdeweerd@users.noreply.github.com>
 * Copyright (C) 2024     Frédéric France             <frederic.france@free.fr>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 */

/**
 *  \file       htdocs/product/stock/massstockmove.php
 *  \ingroup    stock
 *  \brief      Select several products, pick source/target warehouses, and create stock movements.
 */

// Load Dolibarr environment
require '../../../main.inc.php';
require_once DOL_DOCUMENT_ROOT.'/product/class/product.class.php';
require_once DOL_DOCUMENT_ROOT.'/product/stock/class/entrepot.class.php';
require_once DOL_DOCUMENT_ROOT.'/core/class/html.form.class.php';
require_once DOL_DOCUMENT_ROOT.'/core/class/html.formother.class.php';
require_once DOL_DOCUMENT_ROOT.'/fourn/class/fournisseur.commande.class.php';
require_once DOL_DOCUMENT_ROOT.'/product/class/html.formproduct.class.php';
require_once DOL_DOCUMENT_ROOT.'/core/modules/import/import_csv.modules.php';
require_once DOL_DOCUMENT_ROOT.'/core/lib/files.lib.php';
require_once DOL_DOCUMENT_ROOT.'/core/lib/import.lib.php';

/**
 * @var Conf $conf
 * @var DoliDB $db
 * @var HookManager $hookmanager
 * @var Translate $langs
 * @var User $user
 */

$confirm = GETPOST('confirm', 'alpha');
$filetoimport = GETPOST('filetoimport');

// Load translation files required by the page
$langs->loadLangs(array('products', 'stocks', 'orders', 'productbatch'));

// Init Hook
$hookmanager->initHooks(array('massstockmove'));

// Security check
if ($user->socid) {
    $socid = $user->socid;
}
restrictedArea($user, 'produit|service');

// Inputs
$action     = GETPOST('action', 'aZ09');
$id_product = GETPOSTINT('productid');
$id_sw      = GETPOSTINT('id_sw');
$id_tw      = GETPOSTINT('id_tw');
$batch      = GETPOST('batch');
$qty        = GETPOST('qty');
$idline     = GETPOSTINT('idline');

// Pagination (kept minimal here if needed later for lists)
$limit = GETPOSTINT('limit') ? GETPOSTINT('limit') : $conf->liste_limit;
$sortfield = GETPOST('sortfield', 'aZ09comma');
$sortorder = GETPOST('sortorder', 'aZ09comma');
$page = GETPOSTISSET('pageplusone') ? (GETPOSTINT('pageplusone') - 1) : GETPOSTINT("page");
if (empty($page) || $page < 0 || GETPOST('button_search', 'alpha') || GETPOST('button_removefilter', 'alpha')) $page = 0;
if (!$sortfield) $sortfield = 'p.ref';
if (!$sortorder) $sortorder = 'ASC';

// Session list
if (GETPOST('init')) {
    unset($_SESSION['massstockmove']);
}
$listofdata = array();
if (!empty($_SESSION['massstockmove'])) {
    $listofdata = json_decode($_SESSION['massstockmove'], true);
    if (!is_array($listofdata)) $listofdata = array();
}

$error = 0;
$permissiontodelete = $user->hasRight('stock', 'mouvement', 'creer');

/*
 * Actions
 */

// Mode selector
if ($action == 'setmode') {
    $_SESSION['massmove_mode'] = GETPOST('mode', 'alpha');
    header("Location: ".$_SERVER["PHP_SELF"]);
    exit;
}
$mode = !empty($_SESSION['massmove_mode']) ? $_SESSION['massmove_mode'] : 'default';

/* ---- DEFAULT MODE ACTIONS START ---- */
if ($mode == 'default') {

    
/*
 * Actions
 */

if ($action == 'addline' && $user->hasRight('stock', 'mouvement', 'creer')) {
	if (!($id_sw > 0)) {
		//$error++;
		//setEventMessages($langs->trans("ErrorFieldRequired", $langs->transnoentitiesnoconv("WarehouseSource")), null, 'errors');
		if ($id_sw < 0) {
			$id_sw = 0;
		}
	}
	if (!($id_tw > 0)) {
		$error++;
		setEventMessages($langs->trans("ErrorFieldRequired", $langs->transnoentitiesnoconv("WarehouseTarget")), null, 'errors');
	}
	if ($id_sw > 0 && $id_tw == $id_sw) {
		$error++;
		$langs->load("errors");
		setEventMessages($langs->trans("ErrorWarehouseMustDiffers"), null, 'errors');
	}
	if (!($id_product > 0)) {
		$error++;
		setEventMessages($langs->trans("ErrorFieldRequired", $langs->transnoentitiesnoconv("Product")), null, 'errors');
	}
	if (!$qty) {
		$error++;
		setEventMessages($langs->trans("ErrorFieldRequired", $langs->transnoentitiesnoconv("Qty")), null, 'errors');
	}

	// Check a batch number is provided if product need it
	if (!$error) {
		$producttmp = new Product($db);
		$producttmp->fetch($id_product);
		if ($producttmp->hasbatch()) {
			if (empty($batch)) {
				$error++;
				$langs->load("errors");
				setEventMessages($langs->trans("ErrorTryToMakeMoveOnProductRequiringBatchData", $producttmp->ref), null, 'errors');
			}
		}
	}

	// TODO Check qty is ok for stock move. Note qty may not be enough yet, but we make a check now to report a warning.
	// What is more important is to have qty when doing action 'createmovements'
	if (!$error) {
		// Warning, don't forget lines already added into the $_SESSION['massstockmove']
		if ($producttmp->hasbatch()) {
		} else {
		}
	}

	//var_dump($_SESSION['massstockmove']);exit;
	if (!$error) {
		if (count(array_keys($listofdata)) > 0) {
			$id = max(array_keys($listofdata)) + 1;
		} else {
			$id = 1;
		}
		$listofdata[$id] = array('id' => $id, 'id_product' => $id_product, 'qty' => $qty, 'id_sw' => $id_sw, 'id_tw' => $id_tw, 'batch' => $batch);
		$_SESSION['massstockmove'] = json_encode($listofdata);

		//unset($id_sw);
		//unset($id_tw);
		unset($id_product);
		unset($batch);
		unset($qty);
	}
}

if ($action == 'delline' && $idline != '' && $user->hasRight('stock', 'mouvement', 'creer')) {
	if (!empty($listofdata[$idline])) {
		unset($listofdata[$idline]);
	}
	if (count($listofdata) > 0) {
		$_SESSION['massstockmove'] = json_encode($listofdata);
	} else {
		unset($_SESSION['massstockmove']);
	}
}

if ($action == 'createmovements' && $user->hasRight('stock', 'mouvement', 'creer')) {
	$error = 0;

	if (!GETPOST("label")) {
		$error++;
		setEventMessages($langs->trans("ErrorFieldRequired", $langs->transnoentitiesnoconv("MovementLabel")), null, 'errors');
	}

	$db->begin();

	if (!$error) {
		$product = new Product($db);

		foreach ($listofdata as $key => $val) {	// Loop on each movement to do
			$id = $val['id'];
			$id_product = $val['id_product'];
			$id_sw = $val['id_sw'];
			$id_tw = $val['id_tw'];
			$qty = price2num($val['qty']);
			$batch = $val['batch'];
			$dlc = -1; // They are loaded later from serial
			$dluo = -1; // They are loaded later from serial

			if (!$error && $id_sw != $id_tw && is_numeric($qty) && $id_product) {
				$result = $product->fetch($id_product);

				$product->load_stock('novirtual'); // Load array product->stock_warehouse

				// Define value of products moved
				$pricesrc = 0;
				if (!empty($product->pmp)) {
					$pricesrc = (float) $product->pmp;
				}
				$pricedest = $pricesrc;

				//print 'price src='.$pricesrc.', price dest='.$pricedest;exit;

				if (empty($conf->productbatch->enabled) || !$product->hasbatch()) {	// If product does not need lot/serial
					// Remove stock if source warehouse defined
					if ($id_sw > 0) {
						$result1 = $product->correct_stock(
							$user,
							$id_sw,
							(float) $qty,
							1,
							GETPOST("label"),
							$pricesrc,
							GETPOST("codemove")
						);
						if ($result1 < 0) {
							$error++;
							setEventMessages($product->error, $product->errors, 'errors');
						}
					}

					// Add stock
					$result2 = $product->correct_stock(
						$user,
						$id_tw,
						(float) $qty,
						0,
						GETPOST("label"),
						$pricedest,
						GETPOST("codemove")
					);
					if ($result2 < 0) {
						$error++;
						setEventMessages($product->error, $product->errors, 'errors');
					}
				} else {
					$arraybatchinfo = $product->loadBatchInfo($batch);
					if (count($arraybatchinfo) > 0) {
						$firstrecord = array_shift($arraybatchinfo);
						$dlc = $firstrecord['eatby'];
						$dluo = $firstrecord['sellby'];
						//var_dump($batch);
						//var_dump($arraybatchinfo);
						//var_dump($firstrecord);
						//var_dump($dlc);
						//var_dump($dluo); exit;
					} else {
						$dlc = '';
						$dluo = '';
					}

					// Remove stock
					if ($id_sw > 0) {
						$result1 = $product->correct_stock_batch(
							$user,
							$id_sw,
							(float) $qty,
							1,
							GETPOST("label"),
							$pricesrc,
							$dlc,
							$dluo,
							$batch,
							GETPOST("codemove")
						);
						if ($result1 < 0) {
							$error++;
							setEventMessages($product->error, $product->errors, 'errors');
						}
					}

					// Add stock
					$result2 = $product->correct_stock_batch(
						$user,
						$id_tw,
						(float) $qty,
						0,
						GETPOST("label"),
						$pricedest,
						$dlc,
						$dluo,
						$batch,
						GETPOST("codemove")
					);
					if ($result2 < 0) {
						$error++;
						setEventMessages($product->error, $product->errors, 'errors');
					}
				}
			} else {
				// dol_print_error(null,"Bad value saved into sessions");
				$error++;
			}
		}
	}
	//var_dump($_SESSION['massstockmove']);exit;

	if (!$error) {
		unset($_SESSION['massstockmove']);

		$db->commit();
		setEventMessages($langs->trans("StockMovementRecorded"), null, 'mesgs');
		header("Location: ".DOL_URL_ROOT.'/product/stock/list.php'); // Redirect to avoid pb when using back
		exit;
	} else {
		$db->rollback();
		setEventMessages($langs->trans("Error"), null, 'errors');
	}
}

} /* ---- DEFAULT MODE ACTIONS END ---- */

/* ---- BULK MODE ACTIONS START ---- */
if ($mode == 'bulk') {
    // Preview items from source warehouse
    if ($action == 'bulkpreview') {
        $id_sw = GETPOSTINT('id_sw');
        $id_tw = GETPOSTINT('id_tw');

        if ($id_sw > 0 && $id_tw > 0 && $id_sw != $id_tw) {
       $sql = "SELECT ps.fk_product as product_id, ps.reel as qty
        FROM ok_product_stock ps
        WHERE ps.fk_entrepot = ".((int)$id_sw)." AND ps.reel > 0";

$resql = $db->query($sql);
if (!$resql) {
    dol_print_error($db);
} else {
    $_SESSION['massstockmove_bulk'] = array();
    while ($obj = $db->fetch_object($resql)) {
        $_SESSION['massstockmove_bulk'][] = array(
            'id_product' => (int) $obj->product_id,
            'qty'        => (float) $obj->qty,
            'id_sw'      => $id_sw,
            'id_tw'      => $id_tw
        );
    }
    setEventMessages($langs->trans("PreviewLoaded"), null, 'mesgs');
}
            setEventMessages($langs->trans("PreviewLoaded"), null, 'mesgs');
        } else {
            setEventMessages($langs->trans("ErrorWarehouseMustDiffers"), null, 'errors');
        }
    }

    // Execute bulk transfer
    if ($action == 'bulktransfer' && $user->hasRight('stock','mouvement','creer')) {
        $error = 0;
        $db->begin();

        foreach ((array) $_SESSION['massstockmove_bulk'] as $row) {
            $product = new Product($db);
            $product->fetch($row['id_product']);
            $pricesrc = !empty($product->pmp) ? (float) $product->pmp : 0;

            $r1 = $product->correct_stock($user, $row['id_sw'], (float) $row['qty'], 1, $langs->trans("BulkTransfer"), $pricesrc);
            $r2 = $product->correct_stock($user, $row['id_tw'], (float) $row['qty'], 0, $langs->trans("BulkTransfer"), $pricesrc);

            if ($r1 < 0 || $r2 < 0) {
                $error++;
                setEventMessages($product->error, $product->errors, 'errors');
                break;
            }
        }

        if (!$error) {
            $db->commit();
            unset($_SESSION['massstockmove_bulk']);
            setEventMessages($langs->trans("StockMovementRecorded"), null, 'mesgs');
            //header("Location: ".DOL_URL_ROOT.'/product/stock/massstockmove.php'); // stay on bulk page
            //exit;
            
            header("Location: ".$_SERVER['PHP_SELF']."?mode=bulk");
exit;

        } else {
            $db->rollback();
        }
    }
}
/* ---- BULK MODE ACTIONS END ---- */


/* ---- DEFAULT MODE ACTIONS: CSV import ---- */
if ($mode == 'default') {
    if ($action == 'importCSV' && $user->hasRight('stock', 'mouvement', 'creer')) {
        dol_mkdir($conf->stock->dir_temp);
        $fullpath = $conf->stock->dir_temp."/".$user->id.'-csvfiletotimport.csv';
        $resultupload = dol_move_uploaded_file($_FILES['userfile']['tmp_name'], $fullpath, 1);

        if (!(is_numeric($resultupload) && $resultupload > 0)) {
            $error++;
            $langs->load("errors");
            setEventMessages($langs->trans("ErrorFailedToSaveFile"), null, 'errors');
        }

        if (!$error) {
            $importcsv = new ImportCsv($db, 'massstocklist');
            $nblinesrecord = $importcsv->import_get_nb_of_lines($fullpath) - 1;
            $importcsv->import_open_file($fullpath);
            $labelsrecord = $importcsv->import_read_record();

            if ($nblinesrecord < 1) {
                $langs->load("errors");
                setEventMessages($langs->trans("ErrorBadNumberOfLinesMustHaveAtLeastOneLinePlusTitle"), null, 'errors');
            } else {
                $i = 0;
                $data = array();
                $productstatic = new Product($db);
                $warehousestatics = new Entrepot($db);
                $warehousestatict = new Entrepot($db);

                // Loop on each line in CSV file
                while (($i < $nblinesrecord) && !$error) {
                    $newrecord = $importcsv->import_read_record();
                    $data[$i] = $newrecord;

                    if (count($data[$i]) == 1) {
                        unset($data[$i]);
                        $i++;
                        continue;
                    }

                    $tmp_id_sw      = $data[$i][0]['val'];
                    $tmp_id_tw      = $data[$i][1]['val'];
                    $tmp_id_product = $data[$i][2]['val'];
                    $tmp_qty        = $data[$i][3]['val'];
                    $tmp_batch      = $data[$i][4]['val'];

                    // Validate product
                    $result = fetchref($productstatic, $tmp_id_product);
                    if ($result <= 0) {
                        $error++;
                        setEventMessages($langs->trans("ErrorRefNotFound", $tmp_id_product), null, 'errors');
                    }
                    $tmp_id_product = $result;

                    // Validate warehouses
                    $result = fetchref($warehousestatics, $tmp_id_sw);
                    if ($result <= 0 && $tmp_id_sw !== '') {
                        $error++;
                        setEventMessages($langs->trans("ErrorRefNotFound", $tmp_id_sw), null, 'errors');
                    }
                    $tmp_id_sw = $result;

                    $result = fetchref($warehousestatict, $tmp_id_tw);
                    if ($result <= 0) {
                        $error++;
                        setEventMessages($langs->trans("ErrorRefNotFound", $tmp_id_tw), null, 'errors');
                    }
                    $tmp_id_tw = $result;

                    if ($tmp_id_sw > 0 && $tmp_id_tw == $tmp_id_sw) {
                        $error++;
                        setEventMessages($langs->trans("ErrorWarehouseMustDiffers"), null, 'errors');
                    }
                    if (!$tmp_qty) {
                        $error++;
                        setEventMessages($langs->trans("ErrorFieldRequired", $langs->transnoentitiesnoconv("Qty")), null, 'errors');
                    }

                    $i++;
                }

                if (!$error) {
                    foreach ($data as $key => $value) {
                        $listofdata[$key] = array(
                            'id'         => $key,
                            'id_sw'      => $data[$key][0]['val'],
                            'id_tw'      => $data[$key][1]['val'],
                            'id_product' => $data[$key][2]['val'],
                            'qty'        => $data[$key][3]['val'],
                            'batch'      => $data[$key][4]['val']
                        );
                    }
                }
            }
        }

        $_SESSION['massstockmove'] = json_encode($listofdata);
    }
}
/* ---- DEFAULT MODE ACTIONS: CSV import END ---- */
/*
 * View
 */
$form = new Form($db);
$formproduct = new FormProduct($db);

$help_url = 'EN:Module_Stocks_En|FR:Module_Stock|ES:Módulo_Stocks|DE:Modul_Bestände';
$title = $langs->trans('MassMovement');

llxHeader('', $title, $help_url, '', 0, 0, '', '', '', 'mod-product page-stock_massstomove');

print load_fiche_titre($langs->trans("MassStockTransferShort"), '', 'stock');

// Mode selector form
print '<form action="'.$_SERVER["PHP_SELF"].'" method="POST" class="formmodepicker">';
print '<input type="hidden" name="token" value="'.newToken().'">';
print '<input type="hidden" name="action" value="setmode">';
print '<div class="center">';
print '<label><input type="radio" name="mode" value="default" '.($mode=='default'?'checked':'').'> Default item selection</label> &nbsp; ';
print '<label><input type="radio" name="mode" value="bulk" '.($mode=='bulk'?'checked':'').'> Bulk warehouse transfer</label> &nbsp; ';
print '<input type="submit" class="button small" value="Apply">';
print '</div>';
print '</form>';
print '<br>';

/* ---- DEFAULT MODE UI ---- */
if ($mode == 'default') {
    print '<form action="'.$_SERVER["PHP_SELF"].'" method="POST" name="formulaire">';
    print '<input type="hidden" name="token" value="'.newToken().'">';
    print '<input type="hidden" name="action" value="addline">';

    print '<div class="div-table-responsive-no-min">';
    print '<table class="liste noborder centpercent">';
    print '<tr class="liste_titre"><td>Source warehouse</td><td>Target warehouse</td><td>Product</td>';
    if (isModEnabled('productbatch')) print '<td>Batch</td>';
    print '<td>Qty</td><td></td></tr>';

    print '<tr class="oddeven">';
    // Source warehouse dropdown
    print '<td>'.$formproduct->selectWarehouses((isset($id_sw)?$id_sw:''),'id_sw','warehouseopen,warehouseinternal',1,0,0,'',0,0,array(),'minwidth200imp maxwidth200').'</td>';
    // Target warehouse dropdown
    print '<td>'.$formproduct->selectWarehouses((isset($id_tw)?$id_tw:''),'id_tw','warehouseopen,warehouseinternal',1,0,0,'',0,0,array(),'minwidth200imp maxwidth200').'</td>';
    // Product dropdown filtered by source warehouse
    print '<td>';
    
    if ($id_sw) {
       $sql = "SELECT p.rowid, p.ref, p.label, ps.reel
        FROM ok_product p
        INNER JOIN ok_product_stock ps ON ps.fk_product = p.rowid
        WHERE ps.fk_entrepot = ".((int)$id_sw)." AND ps.reel > 0
        ORDER BY p.ref";
        
        $resql = $db->query($sql);
        $options = array();
        while ($obj = $db->fetch_object($resql)) {
            $options[$obj->rowid] = $obj->ref.' - '.$obj->label;
        }
        print $form->selectarray('productid',$options,$id_product,1);
    } else {
        print $form->select_produits((isset($id_product)?$id_product:0),'productid');
    }
    print '</td>';
    // Batch input
    if (isModEnabled('productbatch')) {
        print '<td><input type="text" name="batch" class="flat maxwidth75" value="'.dol_escape_htmltag((isset($batch)?$batch:'')).'"></td>';
    }
    // Qty input
    print '<td class="right"><input type="text" class="flat maxwidth50 right" name="qty" value="'.dol_escape_htmltag((isset($qty)?$qty:'')).'"></td>';
    // Add line button
    print '<td class="right"><input type="submit" class="button" name="addline" value="Add line"></td>';
    print '</tr>';

    // Existing lines rendering
    foreach ($listofdata as $key => $val) {
        // Render each line with product, qty, source, target, batch, and delete option
        // (keep your original rendering logic here)
    }

    print '</table>';
    print '</div>';
    print '</form>';
    print '<br>';
}

/* ---- BULK MODE UI ---- */
if ($mode == 'bulk') {
    print '<form action="'.$_SERVER["PHP_SELF"].'" method="POST" class="formbulk">';
    print '<input type="hidden" name="token" value="'.newToken().'">';
    print '<input type="hidden" name="action" value="bulkpreview">';
    print '<div class="center">';
    print 'Source warehouse: '.$formproduct->selectWarehouses((isset($id_sw)?$id_sw:''),'id_sw','warehouseopen,warehouseinternal');
    print ' &nbsp; ';
    print 'Target warehouse: '.$formproduct->selectWarehouses((isset($id_tw)?$id_tw:''),'id_tw','warehouseopen,warehouseinternal');
    print ' &nbsp; ';
    print '<input type="submit" class="button" value="Preview items">';
    print '</div>';
    print '</form>';
}

// Bulk preview table + Transfer button
if ($mode == 'bulk' && !empty($_SESSION['massstockmove_bulk'])) {
    print '<div class="center"><strong>Items to transfer</strong></div>';
    print '<table class="liste centpercent">';
    print '<tr class="liste_titre"><td>Product</td><td class="right">Qty</td></tr>';
foreach ($_SESSION['massstockmove_bulk'] as $row) {
    $p = new Product($db);
    $p->fetch($row['id_product']);

    $productid   = (int) $p->id;
    $productref  = dol_escape_htmltag($p->ref);
    $productname = dol_escape_htmltag($p->label);
    $qtydisplay  = number_format((float)$row['qty'], 0, '.', ',');

    print '<tr>
        <td>'.$productid.' — '.$productref.' — '.$productname.'</td>
        <td class="right">'.$qtydisplay.'</td>
    </tr>';
}
    print '</table><br>';

    print '<form action="'.$_SERVER["PHP_SELF"].'" method="POST">';
    print '<input type="hidden" name="token" value="'.newToken().'">';
    print '<input type="hidden" name="action" value="bulktransfer">';
    $id_sw = $_SESSION['massstockmove_bulk'][0]['id_sw'];
    $id_tw = $_SESSION['massstockmove_bulk'][0]['id_tw'];
    print '<input type="hidden" name="id_sw" value="'.$id_sw.'">';
    print '<input type="hidden" name="id_tw" value="'.$id_tw.'">';
    print '<div class="center"><input type="submit" class="button" name="bulktransfer" value="Transfer all items"></div>';
    print '</form>';
}

/* ---- Validate all movements button (default mode) ---- */
if ($mode == 'default' && count($listofdata)) {
    print '<form action="'.$_SERVER["PHP_SELF"].'" method="POST" name="formulaire2" class="formconsumeproduce">';
    print '<input type="hidden" name="token" value="'.newToken().'">';
    print '<input type="hidden" name="action" value="createmovements">';

    $codemove = (GETPOSTISSET("codemove") ? GETPOST("codemove", 'alpha') : dol_print_date(dol_now(), '%Y%m%d%H%M%S'));
    $labelmovement = GETPOST("label") ? GETPOST('label') : $langs->trans("MassStockTransferShort").' '.dol_print_date(dol_now(), '%Y-%m-%d %H:%M');

    print '<div class="center">';
    print '<span class="fieldrequired">Inventory code:</span> ';
    print '<input type="text" name="codemove" class="maxwidth300" value="'.dol_escape_htmltag($codemove).'"> &nbsp; ';
    print '<span class="clearbothonsmartphone"></span>';
    print 'Movement label: ';
    print '<input type="text" name="label" class="minwidth300" value="'.dol_escape_htmltag($labelmovement).'"><br>';
    print '<br>';

    print '<div class="center"><input type="submit" class="button" name="valid" value="'.dol_escape_htmltag($buttonrecord).'"></div>';
    print '<br>';
    print '</div>';

    print '</form>';
}

// End of page
llxFooter();
$db->close();


/**
 * Verify if $haystack startswith $needle
 *
 * @param string $haystack
 * @param string $needle
 * @return bool
 */
function startsWith($haystack, $needle)
{
    $length = strlen($needle);
    return substr($haystack, 0, $length) === $needle;
}

/**
 * Fetch object with ref
 *
 * @param CommonObject $static_object
 * @param string $tmp_ref
 * @return int Id of object or <=0 if not found
 */
function fetchref($static_object, $tmp_ref)
{
    if (startsWith($tmp_ref, 'ref:')) {
        $tmp_ref = str_replace('ref:', '', $tmp_ref);
    }
    $static_object->id = 0;
    $static_object->fetch(0, $tmp_ref);
    return $static_object->id;
}
